<?php
// app/Models/ProductRepository.php
namespace App\Models;

use App\Core\Database;
use App\Models\Entities\ProductEntity;

class ProductRepository extends BaseRepository
{
    public function __construct(Database $db)
    {
        // předáme třídu entity do BaseRepository
        parent::__construct($db, 'products', ProductEntity::class);
    }

    /**
     * Vytvoří nový produkt (přijímáme pole z formuláře).
     * Vrací ID nově vloženého záznamu.
     */
    public function create(array $data): int
    {
        $stmt = $this->pdo->prepare(
            "INSERT INTO products (name, category, price, stock, supplier_id) 
             VALUES (:name, :category, :price, :stock, :supplier_id)"
        );
        $stmt->execute([
            'name' => $data['name'],
            'category' => $data['category'],
            'price' => $data['price'],
            'stock' => $data['stock'],
            'supplier_id' => $data['supplier_id'] ?: null
        ]);
        return (int)$this->pdo->lastInsertId();
    }

    /**
     * Aktualizace produktu (pole z formuláře).
     */
    public function update(int $id, array $data): bool
    {
        $stmt = $this->pdo->prepare(
            "UPDATE products SET name=:name, category=:category, price=:price, stock=:stock, supplier_id=:supplier_id WHERE id=:id"
        );
        return $stmt->execute([
            'name' => $data['name'],
            'category' => $data['category'],
            'price' => $data['price'],
            'stock' => $data['stock'],
            'supplier_id' => $data['supplier_id'] ?: null,
            'id' => $id
        ]);
    }

    /**
     * Vrátí pole ProductEntity včetně jména dodavatele (supplier_name).
     */
    public function findAllWithSupplier(): array
    {
        $sql = "SELECT p.*, s.name as supplier_name 
                FROM products p 
                LEFT JOIN suppliers s ON p.supplier_id = s.id 
                ORDER BY p.id DESC";
        $stmt = $this->pdo->query($sql);
        $rows = $stmt->fetchAll();

        // mapovat na entity a nastavit supplier_name, pokud existuje
        $entities = [];
        foreach ($rows as $r) {
            /** @var ProductEntity $entity */
            $entity = $this->mapRowToEntity($r);
            // mapRowToEntity nastaví základní vlastnosti; supplier_name může být dynamická nebo definovaná v entitě
            if (isset($r['supplier_name'])) {
                // pokud má ProductEntity deklarovanou vlastnost supplier_name, bude nastavena,
                // jinak PHP dovolí dynamické přidání (pokud objekt není deklarován jako readonly)
                $entity->supplier_name = $r['supplier_name'];
            }
            $entities[] = $entity;
        }
        return $entities;
    }

    public function findByCategory(string $category)
    {
        $sql = "SELECT p.*, s.name as supplier_name 
                FROM products p 
                LEFT JOIN suppliers s ON p.supplier_id = s.id 
                WHERE p.category = :category
                ORDER BY p.id DESC";
        // $stmt = $this->pdo->query($sql);
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['category' => $category]);
        $rows = $stmt->fetchAll();

        // mapovat na entity a nastavit supplier_name, pokud existuje
        $entities = [];
        foreach ($rows as $r) {
            /** @var ProductEntity $entity */
            $entity = $this->mapRowToEntity($r);
            // mapRowToEntity nastaví základní vlastnosti; supplier_name může být dynamická nebo definovaná v entitě
            if (isset($r['supplier_name'])) {
                // pokud má ProductEntity deklarovanou vlastnost supplier_name, bude nastavena,
                // jinak PHP dovolí dynamické přidání (pokud objekt není deklarován jako readonly)
                $entity->supplier_name = $r['supplier_name'];
            }
            $entities[] = $entity;
        }
        return $entities;
    }
}